<?php

namespace App\Http\Controllers\Auth;

use Carbon\Carbon;
use App\Models\User;
use Illuminate\Http\Request;
use App\Services\CacheService;
use App\Mail\EmailVerification;
use App\Services\CommissionService;
use App\Http\Controllers\Controller;
use App\Models\UserWallet;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use Illuminate\Validation\Rules\Password;

class RegisterController extends Controller
{

    function showRegistrationForm($username = '')
    {
        return view('auth.register', [
            'referralInfo' => User::where('username', $username)->first()
        ]);
    }

    function store(Request $request)
    {
        CacheService::removeAll();
        $request->validate([
            'referral' => ['required', 'exists:App\Models\User,username'],
            'username' => 'required|unique:users|max:20|min:4|regex:/^[a-zA-Z0-9_]+$/u',
            'email' => 'required',
            'password' => ['required', 'confirmed', Password::min(6)],
            'agreement' => '',
        ]);

        $ref = User::where('username', $request->referral)->first();
        if (!User::regFeeVerification($ref)) {
            return redirect()
                ->back()
                ->withInput($request->input())->with('error','This referral "' . ucfirst($request->referral) . '" has not paid the registration fee');
        }

        $config = CacheService::configuration();
        $email_verified_at = null;
        if (!$config || $config->email_verify == "Disabled") {
            $email_verified_at = Carbon::now()->toDateTimeString();
        }

        $user = User::create([
            'referral' => $request->referral,
            'username' => str_replace(' ', '', $request->username),
            'email' => $request->email,
            'password' => Hash::make($request->password),
            'email_verified_at' => $email_verified_at,
            'signup_ip' => request()->ip(),
            'lastseen_ip' => request()->ip(),
            'lastseen_at' => Carbon::now()->toDateTimeString()
        ]);
        UserWallet::create([
            'user_id' => $user->id,
        ]);
        Auth::login($user);

        if ($config && $config->email_verify == "Enabled") {
            $user->update([
                'email_code' => rand(100000, 999999),
            ]);
            Mail::to($user)->send(new EmailVerification($user));

            return redirect()
                ->route('verifyEmail')
                ->with('status', 'A verification has been sent to your email.');
        }

        return redirect()
            ->route('personal.show')
            ->with('status', 'Account created successfully, please enter your personal information');
    }
}
